import UIAbility from "@ohos:app.ability.UIAbility";
import type window from "@ohos:window";
import type { BusinessError } from "@ohos:base";
import hilog from "@ohos:hilog";
import ThemeManager from "@bundle:com.example.myapplication/entry/ets/common/ThemeManager";
import type AbilityConstant from "@ohos:app.ability.AbilityConstant";
import type Want from "@ohos:app.ability.Want";
const TAG = 'EntryAbility';
export default class EntryAbility extends UIAbility {
    /**
     * Ability实例创建时触发 - 应用初始化阶段
     */
    async onCreate(want: Want, launchParam: AbilityConstant.LaunchParam): Promise<void> {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onCreate - 应用实例创建');
        try {
            await ThemeManager.initTheme(this.context);
            hilog.info(0x0000, TAG, '主题管理器初始化完成');
        }
        catch (error) {
            hilog.error(0x0000, TAG, '主题管理器初始化失败: %{public}s', JSON.stringify(error));
        }
    }
    /**
     * 窗口阶段创建 - UI准备阶段
     */
    onWindowStageCreate(windowStage: window.WindowStage): void {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onWindowStageCreate');
        // 关键修复：使用正确的回调函数签名
        windowStage.loadContent('pages/Index', (err: BusinessError) => {
            if (err) {
                hilog.error(0x0000, TAG, '加载页面失败: %{public}s', JSON.stringify(err));
                return;
            }
            // 成功时没有数据返回，只需要记录成功日志
            hilog.info(0x0000, TAG, '页面加载成功');
            // 如果需要获取窗口信息，可以通过其他API单独获取
            this.logWindowInfo(windowStage);
        });
    }
    /**
     * 通过单独的方法获取窗口信息
     */
    private async logWindowInfo(windowStage: window.WindowStage): Promise<void> {
        try {
            const mainWindow = await this.getMainWindow(windowStage);
            if (mainWindow) {
                hilog.info(0x0000, TAG, '主窗口信息获取成功');
                // 可以在这里获取具体的窗口属性
            }
        }
        catch (error) {
            hilog.warn(0x0000, TAG, '获取窗口信息失败: %{public}s', JSON.stringify(error));
        }
    }
    /**
     * 安全地获取主窗口对象
     */
    private getMainWindow(windowStage: window.WindowStage): Promise<window.Window | null> {
        return new Promise((resolve) => {
            windowStage.getMainWindow((err: BusinessError, mainWindow: window.Window) => {
                if (err) {
                    hilog.warn(0x0000, TAG, '获取主窗口失败: %{public}s', JSON.stringify(err));
                    resolve(null);
                    return;
                }
                resolve(mainWindow);
            });
        });
    }
    /**
     * 应用切换到前台时触发
     */
    onForeground(): void {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onForeground - 应用进入前台');
        this.resumeApplicationActivities();
    }
    /**
     * 应用切换到后台时触发
     */
    onBackground(): void {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onBackground - 应用进入后台');
        this.pauseNonCriticalActivities();
    }
    /**
     * 窗口阶段销毁时触发
     */
    onWindowStageDestroy(): void {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onWindowStageDestroy');
        this.cleanupUIResources();
    }
    /**
     * Ability实例销毁时触发
     */
    onDestroy(): void {
        hilog.info(0x0000, TAG, '%{public}s', 'EntryAbility onDestroy');
        this.cleanupAllResources();
    }
    // ==================== 私有方法 ====================
    /**
     * 恢复应用在前台时的活动
     */
    private resumeApplicationActivities(): void {
        // 实现具体的恢复逻辑
        hilog.info(0x0000, TAG, '恢复前台活动');
    }
    /**
     * 暂停非关键后台活动
     */
    private pauseNonCriticalActivities(): void {
        // 实现具体的暂停逻辑
        hilog.info(0x0000, TAG, '暂停非关键活动');
    }
    /**
     * 清理UI相关资源
     */
    private cleanupUIResources(): void {
        // 实现具体的UI资源清理逻辑
        hilog.info(0x0000, TAG, 'UI资源清理完成');
    }
    /**
     * 清理所有应用资源
     */
    private cleanupAllResources(): void {
        // 实现最终的资源清理逻辑
        hilog.info(0x0000, TAG, '所有资源清理完成');
    }
}
